<?php
/**
 * Plugin Name: MiniMaint
 * Plugin URI: https://devinred.com/mini-maint/
 * Description: Minimalist plugin for lightweight maintenance with cookie bypass.
 * Version: 1.0.0
 * Author: devInRed
 * Author URI: https://devinred.com/
 * License: GPL v2 or later
 * License URI: https://www.gnu.org/licenses/gpl-2.0.html
 * Text Domain: minimaint
 */

// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

class MiniMaint {
    
    private $option_name = 'minimaint_settings';
    private $default_cookie_name = 'minimaint_bypass';
    
    public function __construct() {
        // Admin menu
        add_action( 'admin_menu', array( $this, 'add_admin_menu' ) );
        add_action( 'admin_init', array( $this, 'register_settings' ) );
        
        // Maintenance mode logic
        add_action( 'template_redirect', array( $this, 'maintenance_mode' ), 1 );
        
        // AJAX handler for cookie generation
        add_action( 'wp_ajax_minimaint_generate_cookie', array( $this, 'generate_cookie_ajax' ) );
        
        // Remove header/footer on maintenance page
        add_action( 'template_redirect', array( $this, 'maybe_remove_header_footer' ) );
    }
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_options_page(
            __( 'MiniMaint Settings', 'minimaint' ),
            __( 'minimaint', 'minimaint' ),
            'manage_options',
            'minimaint',
            array( $this, 'settings_page' )
        );
    }
    
    /**
     * Register settings
     */
    public function register_settings() {
        register_setting(
            'minimaint_options',
            $this->option_name,
            array( $this, 'sanitize_settings' )
        );
    }
    
    /**
     * Sanitize settings
     */
    public function sanitize_settings( $input ) {
        $output = array();
        
        $output['enabled'] = isset( $input['enabled'] ) ? 1 : 0;
        $output['page_id'] = isset( $input['page_id'] ) ? absint( $input['page_id'] ) : 0;
        $output['cookie_name'] = isset( $input['cookie_name'] ) ? sanitize_key( $input['cookie_name'] ) : $this->default_cookie_name;
        $output['cookie_value'] = isset( $input['cookie_value'] ) ? sanitize_text_field( $input['cookie_value'] ) : '';
        $output['hide_header_footer'] = isset( $input['hide_header_footer'] ) ? 1 : 0;
        
        return $output;
    }
    /**
     * Get settings
     */
    private function get_settings() {
        return wp_parse_args(
            get_option( $this->option_name, array() ),
            array(
                'enabled' => 0,
                'page_id' => 0,
                'cookie_name' => $this->default_cookie_name,
                'cookie_value' => '',
                'hide_header_footer' => 0
            )
        );
    }
    
    /**
     * Settings page
     */
    public function settings_page() {
        if ( ! current_user_can( 'manage_options' ) ) {
            return;
        }
        
        // Save settings
        if ( isset( $_POST['minimaint_nonce'], $_POST['minimaint'] ) && wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['minimaint_nonce'] ) ), 'minimaint_save' ) ) {
            // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Array is sanitized in sanitize_settings method
            $settings = $this->sanitize_settings( wp_unslash( $_POST['minimaint'] ) );
            update_option( $this->option_name, $settings );
            echo '<div class="notice notice-success"><p>' . esc_html__( 'Settings saved.', 'minimaint' ) . '</p></div>';
        }
        
        $settings = $this->get_settings();
        $pages = get_pages();
        
        ?>
        <div class="wrap">
            <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
            
            <form method="post" action="">
                <?php wp_nonce_field( 'minimaint_save', 'minimaint_nonce' ); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="minimaint_enabled"><?php esc_html_e( 'Maintenance Mode', 'minimaint' ); ?></label>
                        </th>
                        <td>
                            <label>
                                <input type="checkbox" name="minimaint[enabled]" id="minimaint_enabled" value="1" <?php checked( $settings['enabled'], 1 ); ?>>
                                <?php esc_html_e( 'Enable maintenance mode', 'minimaint' ); ?>
                            </label>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="minimaint_page_id"><?php esc_html_e( 'Maintenance Page', 'minimaint' ); ?></label>
                        </th>
                        <td>
                            <select name="minimaint[page_id]" id="minimaint_page_id" class="regular-text">
                                <option value="0"><?php esc_html_e( 'Select a page', 'minimaint' ); ?></option>
                                <?php foreach ( $pages as $page ) : ?>
                                    <option value="<?php echo esc_attr( $page->ID ); ?>" <?php selected( $settings['page_id'], $page->ID ); ?>>
                                        <?php echo esc_html( $page->post_title ); ?>
                                    </option>
                                <?php endforeach; ?>
                            </select>
                            <p class="description"><?php esc_html_e( 'Select the page to display during maintenance.', 'minimaint' ); ?></p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="minimaint_cookie_name"><?php esc_html_e( 'Cookie Name', 'minimaint' ); ?></label>
                        </th>
                        <td>
                            <input type="text" name="minimaint[cookie_name]" id="minimaint_cookie_name" value="<?php echo esc_attr( $settings['cookie_name'] ); ?>" class="regular-text">
                            <p class="description"><?php esc_html_e( 'Name of the cookie to bypass maintenance mode.', 'minimaint' ); ?></p>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="minimaint_cookie_value"><?php esc_html_e( 'Cookie Value', 'minimaint' ); ?></label>
                        </th>
                        <td>
                            <input type="text" name="minimaint[cookie_value]" id="minimaint_cookie_value" value="<?php echo esc_attr( $settings['cookie_value'] ); ?>" class="regular-text" readonly>
                            <button type="button" id="minimaint_generate_cookie" class="button"><?php esc_html_e( 'Generate New Value', 'minimaint' ); ?></button>
                            <p class="description"><?php esc_html_e( 'Random value for the bypass cookie.', 'minimaint' ); ?></p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">
                            <?php esc_html_e( 'Bypass Link', 'minimaint' ); ?>
                        </th>
                        <td>
                            <?php if ( ! empty( $settings['cookie_value'] ) ) : ?>
                                <input type="text" id="minimaint_bypass_link" value="<?php echo esc_url( $this->get_bypass_link( $settings ) ); ?>" class="regular-text" readonly>
                                <button type="button" id="minimaint_copy_link" class="button"><?php esc_html_e( 'Copy Link', 'minimaint' ); ?></button>
                                <p class="description"><?php esc_html_e( 'Share this link to allow access during maintenance.', 'minimaint' ); ?></p>
                            <?php else : ?>
                                <p class="description"><?php esc_html_e( 'Generate a cookie value to get the bypass link.', 'minimaint' ); ?></p>
                            <?php endif; ?>
                        </td>
                    </tr>
                    
                    <tr>
                        <th scope="row">
                            <label for="minimaint_hide_header_footer"><?php esc_html_e( 'Layout', 'minimaint' ); ?></label>
                        </th>
                        <td>
                            <label>
                                <input type="checkbox" name="minimaint[hide_header_footer]" id="minimaint_hide_header_footer" value="1" <?php checked( $settings['hide_header_footer'], 1 ); ?>>
                                <?php esc_html_e( 'Hide header and footer on maintenance page', 'minimaint' ); ?>
                            </label>
                            <p class="description"><?php esc_html_e( 'Display only the page content without theme header/footer.', 'minimaint' ); ?></p>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button(); ?>
            </form>
            
            <hr>
            
            <div class="minimaint-info" style="margin-top: 20px; padding: 15px; background: #f9f9f9; border-left: 4px solid #2271b1;">
                <h2><?php esc_html_e( 'About MiniMaint', 'minimaint' ); ?></h2>
                <p><?php esc_html_e( 'Lightweight maintenance mode plugin with easy cookie-based bypass system.', 'minimaint' ); ?></p>
                <p>
                    <strong><?php esc_html_e( 'Version:', 'minimaint' ); ?></strong> 1.0.0<br>
                    <strong><?php esc_html_e( 'Website:', 'minimaint' ); ?></strong> <a href="https://devinred.com/mini-maint/" target="_blank">devinred.com/mini-maint</a><br>
                </p>
            </div>
        </div>
        
        <script>
        jQuery(document).ready(function($) {
            $('#minimaint_generate_cookie').on('click', function() {
                var button = $(this);
                button.prop('disabled', true);
                
                $.post(ajaxurl, {
                    action: 'minimaint_generate_cookie',
                    nonce: '<?php echo esc_attr( wp_create_nonce( 'minimaint_generate_cookie' ) ); ?>'
                }, function(response) {
                    if (response.success) {
                        $('#minimaint_cookie_value').val(response.data.cookie_value);
                        if (response.data.bypass_link) {
                            $('#minimaint_bypass_link').val(response.data.bypass_link);
                        }
                    }
                    button.prop('disabled', false);
                });
            });
            
            $('#minimaint_copy_link').on('click', function() {
                var link = $('#minimaint_bypass_link');
                link.select();
                document.execCommand('copy');
                
                var button = $(this);
                var originalText = button.text();
                button.text('<?php esc_html_e( 'Copied!', 'minimaint' ); ?>');
                setTimeout(function() {
                    button.text(originalText);
                }, 2000);
            });
        });
        </script>
        <?php
    }
    
    /**
     * Generate bypass link
     */
    private function get_bypass_link( $settings ) {
        return add_query_arg(
            'minimaint_bypass',
            $settings['cookie_value'],
            home_url()
        );
    }
    
    /**
     * AJAX: Generate cookie value
     */
    public function generate_cookie_ajax() {
        check_ajax_referer( 'minimaint_generate_cookie', 'nonce' );
        
        if ( ! current_user_can( 'manage_options' ) ) {
            wp_send_json_error();
        }
        
        $settings = $this->get_settings();
        $cookie_value = wp_generate_password( 32, false, false );
        
        wp_send_json_success( array(
            'cookie_value' => $cookie_value,
            'bypass_link' => add_query_arg(
                'minimaint_bypass',
                $cookie_value,
                home_url()
            )
        ) );
    }
    
    /**
     * Remove header and footer on maintenance page if enabled
     */
    public function maybe_remove_header_footer() {
        $settings = $this->get_settings();
        
        if ( ! $settings['hide_header_footer'] ) {
            return;
        }
        
        if ( ! $settings['page_id'] ) {
            return;
        }
        
        if ( is_page( $settings['page_id'] ) ) {
            remove_action( 'wp_head', '_wp_render_title_tag', 1 );
            add_filter( 'template_include', array( $this, 'blank_template' ) );
        }
    }
    
    /**
     * Blank template for maintenance page
     */
    public function blank_template( $template ) {
        $settings = $this->get_settings();
        
        if ( ! is_page( $settings['page_id'] ) ) {
            return $template;
        }
        
        return plugin_dir_path( __FILE__ ) . 'template-blank.php';
    }
    
    /**
     * Maintenance mode logic
     */
    public function maintenance_mode() {
        $settings = $this->get_settings();
        
        // Skip if maintenance is not enabled
        if ( ! $settings['enabled'] ) {
            return;
        }
        
        // Skip if no page is selected
        if ( ! $settings['page_id'] ) {
            return;
        }
        
        // Skip for administrators only
        if ( current_user_can( 'manage_options' ) ) {
            return;
        }
        
        // Skip for admin pages
        if ( is_admin() ) {
            return;
        }
        
        // Check if user has bypass cookie
        if ( ! empty( $settings['cookie_value'] ) && isset( $_COOKIE[ $settings['cookie_name'] ] ) ) {
            if ( $_COOKIE[ $settings['cookie_name'] ] === $settings['cookie_value'] ) {
                return;
            }
        }
        
        // Set bypass cookie if link is clicked
        // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Bypass link uses cookie value verification instead of nonce for non-logged users
        $bypass_value = isset( $_GET['minimaint_bypass'] ) ? sanitize_text_field( wp_unslash( $_GET['minimaint_bypass'] ) ) : '';
        if ( ! empty( $bypass_value ) && $bypass_value === $settings['cookie_value'] ) {
            setcookie(
                $settings['cookie_name'],
                $settings['cookie_value'],
                time() + ( 30 * DAY_IN_SECONDS ),
                COOKIEPATH,
                COOKIE_DOMAIN,
                is_ssl(),
                true
            );
            wp_safe_redirect( home_url() );
            exit;
        }
        
        // Skip if already on maintenance page
        if ( is_page( $settings['page_id'] ) ) {
            return;
        }
        
        // Redirect to maintenance page
        wp_safe_redirect( get_permalink( $settings['page_id'] ) );
        exit;
    }
}

// Initialize plugin
new MiniMaint();
